/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

static const char __idstring[] = "@(#)$Id: mx_isend.c,v 1.15 2005/08/18 21:25:26 gallatin Exp $";

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <netinet/in.h>
#include <unistd.h>

#include "mx_auto_config.h"
#include "myriexpress.h"
#include "internal.h"
#include "tcp.h"
#include "packet.h"
#include "rdma.h"

/*
  mx_return_t
  mx_isend(mx_endpoint_t endpoint,
  mx_segment_t *segments_list,
  uint32_t segments_count,
  mx_endpoint_addr_t dest_endpoint,
  uint64_t match_data,
  void *callback_arg,
  mx_request_t *handle)
*/

mx_return_t 
mx_isend(mx_endpoint_t endpoint, 
         mx_segment_t *segments_list, 
         uint32_t segments_count, 
         mx_endpoint_addr_t destination, 
	 uint64_t match_info,
         void *context, 
         mx_request_t *request)
{
  return mx_isend_with_type(endpoint, segments_list, segments_count,
                            destination, match_info,
                            context, request, MX_SR_TYPE_SEND,
                            0, 0, 0);
}


/*
  mx_return_t
  mx_isend_with_type(mx_endpoint_t endpoint,
  mx_segment_t *segments_list,
  uint32_t segments_count,
  mx_endpoint_addr_t dest_address,
  uint64_t match_data,
  void *callback_arg,
  mx_request_t *handle,
  mx_post_type_t type)
*/

mx_return_t 
mx_isend_with_type(mx_endpoint_t endpoint, 
                   mx_segment_t *segments_list, 
                   uint32_t segments_count, 
                   mx_endpoint_addr_t destination,
		   uint64_t match_info,
                   void *context, 
                   mx_request_t *request,
                   mx_sr_type_t type,
                   uint32_t rdma_handle,
                   uint32_t rdma_offset,
                   uint32_t rdma_status)
{
  struct mx_endpoint *pp;
  struct mx_post *post;
  struct mx_lib_send *send;
  struct mx_lib_address *x;

  /* XXX address_t -> endpoint_t */
  pthread_mutex_lock(&Mx_tx_lock);
    
  pp = (struct mx_endpoint *)endpoint;
    
  /* Build a post descriptor */
  post = mx_new_post(endpoint, MX_POST_TYPE_SEND, context);
  if (post == NULL) {
    pthread_mutex_unlock(&Mx_tx_lock);
    return MX_NO_RESOURCES;
  }

  /* fill in send-specific fields */
  send = &post->ts.send;
  send->length = mx_segment_list_len(segments_list, segments_count);

  send->seg_list = mx_new_segment_list(segments_list, segments_count);
  if (send->seg_list == NULL) {
    mx_destroy_post(post);
    pthread_mutex_unlock(&Mx_tx_lock);
    return MX_NO_RESOURCES;
  }

  send->seg_cnt = segments_count;
  send->dest = destination;
  send->match_info = match_info;
  send->type = type;
  /* this is not RDMA */
  send->rdma_handle = rdma_handle;
  send->rdma_offset = rdma_offset;
  send->rdma_status = rdma_status;

  /* queue the send as pending */
  MX_LIST_INSERT(&pp->send_list, post);
  x = (struct mx_lib_address *)&destination;
    
  MX_DEBUG_PRINT(MX_DEBUG_TCP_LIB,("posting send(0x%08x%08x) %d bytes to 0x%08x:%d\n",
				   MX_U32(match_info), MX_L32(match_info),
            send->length,
	   (int)ntohl(x->ipaddr_n), ntohs(x->ipport_n)));
    
  /*
   * Try to perform the send now
   */
  /* mx_run_send_queue(pp); */
    
  MX_DEBUG_PRINT(MX_DEBUG_TCP_LIB,("post done\n"));
    
  /* return send handle */
  *request = (mx_request_t) post;
  pthread_cond_signal(&Mx_send_cond);
  pthread_mutex_unlock(&Mx_tx_lock);

  return MX_SUCCESS;
}

uint32_t
mx_try_send(struct mx_post *post)
{
  int s;
  int i;
  int n;
  int rc;
  struct mx_lib_send *send;
  struct mx_send_hdr snd_hdr;
  struct mx_lib_address *x;
  mx_pkt_type_t pkt_type;
    
  /* get ptr to send struct */
  send = &post->ts.send;

  x = (struct mx_lib_address *)&send->dest;
  MX_DEBUG_PRINT(MX_DEBUG_TCP_LIB,("trying send %d bytes to 0x%08x:%d\n", send->length,
				   (int)ntohl(x->ipaddr_n), ntohs(x->ipport_n)));

  /* check if RDMA is allowed */

  if(send->type == MX_SR_TYPE_PUT){
    int i;
    for (i = 0; i < send->seg_cnt; i++){
      if (!mx_rdma_allowed(MX_UINT64(send->seg_list[i].segment_ptr),
                           send->seg_list[i].segment_length,
                           MX_RDMA_READ)){
        post->status.code = MX_STATUS_REJECTED;
        MX_DEBUG_PRINT(MX_DEBUG_TCP_LIB,("Read RDMA not allowed\n"));
        return 1;
      }
    }
  }


  /* See if we can connect to this address */
  /* rx thread must not modify the data structure */
  pthread_mutex_lock(&Mx_po_lock);
  s = get_sock_for_address(post->endpoint, send->dest);
  pthread_mutex_unlock(&Mx_po_lock);

  /* if no connection (yet), return unsuccessful */
  if (s == -1) {
    return 0;
  }
    
  /* got connected, so perform the send */
  if (post->type != MX_POST_TYPE_SEND){
    MX_DEBUG_PRINT(MX_DEBUG_TCP_LIB,("hey, this isn't a send!!!\n"));
    exit(1);
  }

  switch (send->type) {
  case MX_SR_TYPE_SEND:
    pkt_type = MX_PKT_TYPE_SEND;
    break;
  case MX_SR_TYPE_ISSEND:
    pkt_type = MX_PKT_TYPE_ISSEND;
    break;
  case MX_SR_TYPE_BARRIER:
    pkt_type = MX_PKT_TYPE_BARRIER;
    break;
  case MX_SR_TYPE_BCAST:
    pkt_type = MX_PKT_TYPE_BCAST;
    break;
  case MX_SR_TYPE_BARRIER_ACK:
    pkt_type = MX_PKT_TYPE_BARRIER_ACK;
    break;
  case MX_SR_TYPE_ISSEND_ACK:
    pkt_type = MX_PKT_TYPE_ISSEND_ACK;
    break;
  case MX_SR_TYPE_PUT:
    pkt_type = MX_PKT_TYPE_PUT;
    break;
  case MX_SR_TYPE_PUT_ACK:
    pkt_type = MX_PKT_TYPE_PUT_ACK;
    break;
  case MX_SR_TYPE_GET:
    pkt_type = MX_PKT_TYPE_GET;
    break;
  case MX_SR_TYPE_GET_DATA:
    pkt_type = MX_PKT_TYPE_GET_DATA;
    break;
  default:
    MX_DEBUG_PRINT(MX_DEBUG_TCP_LIB,("unknown send post type %d\n", send->type));
    exit(1);
    break;
  }
    
  /* first a header */
  mx_send_header(s, pkt_type, post->endpoint,
                 send->length + sizeof (snd_hdr));
    
  /* then a send sub-header */
  snd_hdr.match_a = htonl(send->match_info >> 32);
  snd_hdr.match_b = htonl(send->match_info & 0xffffffff);
  snd_hdr.request = (mx_request_t)post;
  snd_hdr.rdma_handle = send->rdma_handle;
  snd_hdr.rdma_offset = send->rdma_offset;
  snd_hdr.rdma_status = send->rdma_status;

  rc = mx_sock_write(s, &snd_hdr, sizeof(snd_hdr));
    
  /* then the body of the message */
  for (i=0; i<send->seg_cnt; ++i) {
    n = mx_sock_write(s,
                      send->seg_list[i].segment_ptr,
                      send->seg_list[i].segment_length);
    if (n < 0){
      perror("write error");
      exit(1);
    }
  }
  
  MX_DEBUG_PRINT(MX_DEBUG_TCP_LIB,("completed send, len = %d, handle = 0x%p\n", send->length, post));
    
  /* send successful! */
  return 1;
}
